{
 Designer: originally uploaded by 100115.1155@compuserve.com,
           amendments by Terry Pearson (71715.1647@compuserve.com),
           amendments by Craig Ward (100554.2072@compuserve.com)

 Date:     9/5/96
 Version:  (Beta)


 Function: Component that handles printing of a Table's data.


 Calling:  There are two executable methods:
            [1] Print               - spools data straight to printer
            [2] PrintDialog         - initiates a TPrintDialog

 Notes:    Points to note:

            [1] the 'print quality' property for the TPrintDialog is not taken into
                account when printing (it would be interesting to note how this is achieved
                - my guess is that it's through amendment's to the canvas's font\brush\pen)
            [2] the date label should be set via the application - for example:
                 cwDBPrint1.DateLabel := DateTimeToStr(date);

*******************************************************************************}
unit Cwdbprn;

interface

uses
 SysUtils,  WinTypes,  WinProcs,  Messages,  Classes,  Graphics,  Controls,
 Forms,  Dialogs, DBGrids, DB, DBTables;

const
 MaxPages = 100; {maximum page number}
 MaxCols = 100;  {max column number}


type
 {custom type - page-number position}
 TPageNumberPos = (pnNone,  pnTopLeft, pnTopCenter, pnTopRight, pnBotLeft, pnBotCenter, pnBotRight);


  TcwDBPrint = class(TComponent)
  private
    { Private declarations }
    tmpFile: Text;
    tmpFileName : TFileName;
    FTable: TTable;
    FHeaderAlign: TAlignment;
    FLinesFont: TFont;          {record font}
    FHeaderFont: TFont;         {header\footer font}
    FFieldFont: TFont;          {fieldname font}
    FTitleFont: TFont;          {title font}
    FPageNLabel: string;
    FDateLabel: string;
    FPageNPos: TPageNumberPos;
    FDatePos: TPageNumberPos;
    FPrintFileName: string;
    FHeader: string;
    FPrintMgrTitle: string;
    FirstRecordY: longint;
    LinesWidth: longint;
    LinesHeight: longint;
    RecCounter: longint;
    FToPrint: boolean;
    tmpPageNo: longint;
    FFromPage: longint;
    FToPage: longint;
    NPositions: integer;
    FTopMargin: integer;
    FBottomMargin: integer;
    FLeftMargin: integer;
    FRightMargin: integer;
    Positions: array[1..MaxCols] of longint;
    FColLines: boolean;
    FRowLines: boolean;
    FBorder: boolean;
    FHorizGap: integer;
    FVertGap: integer;
    procedure WriteLineScreen(const S: string);
    procedure SetTable(value: TTable);
    procedure SetFieldFont(Value: TFont);
    procedure SetHeaderFont(Value: TFont);
    procedure SetLinesFont(Value: TFont);
    procedure SetPrintMgrTitle(const S: string);
    procedure SetTitleFont(value: TFont);
    function GetPrintMgrTitle: string;
    function OpenTextForWrite: boolean;
    function ScreenWidth(tmp: TField): longint;
    function TitleWidth(const S: string): longint;
    function TitleHeight: longint;
    procedure CalculatePositions;
    function SetAlign(align:TAlignment; Left, Right: longint): longint;
    function SetPagePosX(PagePos: TPageNumberPos; Left, Right: longint): longint;
    function SetPagePosY(PagePos: TPageNumberPos; Top, Bottom: longint): longint;
    function PrepareAlign(Field: TField; Col: integer; Heading: boolean): longint;
    procedure WriteHeaderToPrinter;
    procedure WriteLabelToPrinter(PosY: longint);
    procedure WriteRecordToPrinter;
    procedure WriteHeader;
    procedure WriteRecord;
    procedure PageJump;
    function RealWidth: longint;
    function AllPageFilled: boolean;
  protected
    { Protected declarations }
    procedure SetName(const Value: TComponentName); override;
  public
    { Public declarations }
    constructor Create(AOwner:TComponent); override;
    destructor Destroy; override;
    procedure Print;
    procedure PrintDialog;
  published
   { Published declarations }
    property LeftMargin: integer read FLeftMargin write FLeftMargin;
    property TopMargin: integer read FTopMargin write FTopMargin;
    property RightMargin: integer read FRightMargin write FRightMargin;
    property BottomMargin: integer read FBottomMargin write FBottomMargin;
    property FieldNameFont: TFont read FFieldFont write SetFieldFont;
    property HeaderFooterFont: TFont read FHeaderFont write SetHeaderFont;
    property RecordFont: TFont read FLinesFont write SetLinesFont;
    property TitleFont: TFont read FTitleFont write SetTitleFont;
    property Table: TTable read FTable write SetTable;
    property PrintMgrTitle: string read GetPrintMgrTitle write SetPrintMgrTitle;
    property Header: string read FHeader write FHeader;
    property HeaderAlignment: TAlignment read FHeaderAlign write FHeaderAlign;
    property PrintToFile: boolean read FToPrint write FToPrint;
    property PrintFileName: string read FPrintFileName write FPrintFileName;
    property FromPage: longint read FFromPage write FFromPage;
    property ToPage: longint read FToPage write FToPage;
    property Border: boolean read FBorder write FBorder;
    property ColLines: boolean read FColLines write FColLines;
    property RowLines: boolean read FRowLines write FRowLines;
    property HorizontalGap: integer read FHorizGap write FHorizGap;
    property VerticalGapPct: integer read FVertGap write FVertGap;
    property PageNumberPos: TPageNumberPos read FPageNPos write FPageNPos;
    property PageNumberLabel: string read FPageNLabel write FPageNLabel;
    property DatePos: TPageNumberPos read FDatePos write FDatePos;
    property DateLabel: string read FDateLabel write FDateLabel;
  end;


{VCL register}
procedure Register;


implementation

uses
 Printers;


{***math routines*************************************************************}

{find max value}
function Max(a, b: longint): longint;
begin
 if a > b then
  Result := a
 else
  Result := b;
end;


{find scale}
function Scale(Value: longint; Pct: integer): longint;
begin

 if Pct > 100 then
  Pct := 100
 else
  if Pct < 0 then
   Pct := 0;

 if Pct = 0 then
  Result := Value
 else
  Result := Value + MulDiv(Value, Pct, 100);

end;

{find central position, vertically}
function CenterY(PosY, TextHt, Pct: longint): longint;
begin
 Result := PosY + (Scale(TextHt, Pct) - TextHt) div 2;
end;


{find screen width}
function TcwDBPrint.ScreenWidth(tmp:TField): longint;
begin
 Result := Max(tmp.DisplayWidth, Length(tmp.DisplayLabel));
end;

{find title width}
function TcwDBPrint.TitleWidth(const S: string): longint;
var
 tmpFont: TFont;
begin
 with Printer.Canvas do
  begin
   tmpFont := TFont.Create;
   tmpFont.Assign(Font); {store printer's font}
   Font.Assign(FTitleFont);
   Result := TextWidth(s);
   Font.Assign(tmpFont);
   tmpFont.Free;
  end;
end;

{find title height in title font}
function TcwDBPrint.TitleHeight: longint;
var
 tmpFont: TFont;
 li: longint;
begin
 with Printer.Canvas do
  begin
   tmpFont := TFont.Create;
   tmpFont.Assign(Font); {store printer's font}
   Font.Assign(FTitleFont);
   li := Scale(TextHeight('M'), FVertGap);
   Font.Assign(tmpFont);
   tmpFont.Free;
   result := li;
  end;
end;


{calculate positions}
procedure TcwDBPrint.CalculatePositions;
var
 ColWidth, t: longint;
begin

 NPositions := 0;

 if FBorder then
  Positions[1] := 1
 else
  Positions[1] := 0;

 {store field display widths in array}
 with FTable do
  for t := 0 to FieldCount - 1 do
   with Fields[t] do
    if Visible then
     begin
      inc(NPositions);
      ColWidth := Max(TitleWidth(Fields[t].DisplayLabel), (LinesWidth * Fields[t].DisplayWidth));
      Positions[NPositions + 1] := Positions[NPositions] + ColWidth + FHorizGap;
     end;

end;


{real width}
function TcwDBPrint.RealWidth: longint;
begin
 Result := Printer.PageWidth - FLeftMargin - FRightMargin;
end;


{***I/O Routines****************************************************************}

{write}
procedure TcwDBPrint.WriteLineScreen(const S: string);
begin
 if (tmpPageNo >= FFromPage) and (tmpPageNo <= FToPage) then Writeln(tmpFile, S);
end;

{open file for writing (used to print-to-file)}
function TcwDBPrint.OpenTextForWrite: boolean;
begin
 if tmpFileName <> '' then
  begin
   {$I-}
   AssignFile(tmpFile, tmpFileName);
   rewrite(tmpFile);
   {$I+}
   Result := (ioresult = 0);
  end
 else
  Result := false;
end;



{***component preferences*******************************************************}


{set Table}
procedure TcwDBPrint.SetTable(value: TTable);
begin
 if value <> FTable then
  begin
   FTable := Value;
  end;
end;


{register}
procedure Register;
begin
 RegisterComponents('cw_apps', [TcwDBPrint]);
end;


{constructor}
constructor TcwDBPrint.Create(AOwner:TComponent);
begin
 inherited Create(AOwner);

  {allocate resources for fields}
  FTitleFont := TFont.Create;
  FHeaderFont := TFont.Create;
  FLinesFont := TFont.Create;
  FFieldFont := TFont.create;

  {defaults}
  RecCounter := 0;
  FHorizGap := 10;
  FVertGap := 20;
  FTopMargin := 40;
  FBottomMargin := 40;
  FLeftMargin := 40;
  FRightMargin := 40;
  FFromPage := 1;
  FToPage := MaxPages;
  FBorder := True;
  FColLines := True;
  FHeaderAlign := taCenter;
  FPageNPos := pnBotCenter;
  FPageNLabel := 'Page ';
  FDatePos := pnBotLeft;

end;


{destructor}
destructor TcwDBPrint.Destroy;
begin
 {free resources allocated to fields}
 FTitleFont.Free;
 FHeaderFont.Free;
 FLinesFont.Free;
 FFieldFont.Free;
 inherited Destroy;
end;


{set field}
procedure TcwDBPrint.SetTitleFont(Value: TFont);
begin
 FTitleFont.Assign(Value);
end;


{set field}
procedure TcwDBPrint.SetFieldFont(Value: TFont);
begin
 FFieldFont.Assign(Value);
end;


{set field}
procedure TcwDBPrint.SetHeaderFont(Value: TFont);
begin
 FHeaderFont.Assign(Value);
end;

{set field}
procedure TcwDBPrint.SetLinesFont(Value: TFont);
begin
 FLinesFont.Assign(Value);
end;


{set field}
procedure TcwDBPrint.SetPrintMgrTitle(const S: string);
begin
 FPrintMgrTitle := S;
end;

{set field}
function TcwDBPrint.GetPrintMgrTitle: string;
begin
 Result := FPrintMgrTitle;
end;

{set field}
procedure TcwDBPrint.SetName(const Value: TComponentName);
var
 ChangeText: Boolean;
begin
  ChangeText := (Name = FPrintMgrTitle) and ((Owner = nil)
                or not (Owner is TcwDBPrint) or
                not (csLoading in TcwDBPrint(Owner).ComponentState));
  inherited SetName(Value);
  if ChangeText then FPrintMgrTitle := Value;
end;




{***printing routines**********************************************************}

{set alignment for text - return 'X' canvas position according to field alignment,
 using API function "SetTextAlign"}
function TcwDBPrint.SetAlign(align: TAlignment; Left, Right: longint): longint;
begin
 with Printer.Canvas do
  begin
   {switch on alignment}
   case Align of
    taLeftJustify: {left}
     begin
      SetTextAlign(Handle, TA_LEFT);
      result := Left + FHorizGap;
     end;
    taRightJustify: {right}
     begin
      SetTextAlign(Handle, TA_RIGHT);
      result := Right - FHorizGap;
     end;
    taCenter: {centre}
     begin
      SetTextAlign(Handle, TA_CENTER);
      result := Left + Round((Right - Left) / 2);
     end;
    end;
  end;
end;


{set horizontal page position - return 'X' canvas position according to page-number
 positioning using API function "SetTextAlign"}
function TcwDBPrint.SetPagePosX(PagePos: TPageNumberPos; Left, Right: longint): longint;
begin
 with Printer.Canvas do
  begin
   {switch on page-number position}
   case PagePos of
    pnTopLeft, pnBotLeft:
     begin
      SetTextAlign(Handle, TA_LEFT);
      result := Left + FHorizGap;
     end;
    pnTopRight, pnBotRight:
     begin
      SetTextAlign(Handle, TA_RIGHT);
      result := Right - FHorizGap;
     end;
    pnTopCenter, pnBotCenter:
     begin
      SetTextAlign(Handle, TA_CENTER);
      result := Left + Round((Right - Left)/2);
     end;
   end;
  end;
end;


{set vertical page position - return 'Y' canvas position according to page-number
 positioning using API function "SetTextAlign"}
function TcwDBPrint.SetPagePosY(PagePos: TPageNumberPos; Top, Bottom: longint): longint;
begin
 {switch on page-number position}
 case PagePos of
  pnBotLeft, pnBotCenter, pnBotRight:
   begin
    result := Bottom;
   end;
  else
   result := Top;
 end;
end;


{prepare alignment - calls SetAlign, and returns 'X' position on canvas. Note, in case of
 heading, default to central position}
function TcwDBPrint.PrepareAlign(Field:TField; Col:integer; Heading: boolean): longint;
begin
 case Heading of
  false: {use field alignment to align text}
   Result := SetAlign(Field.Alignment, Positions[col], Positions[col + 1]);
  true:  {default to central alignment}
   Result := SetAlign(taCenter,Positions[col],Positions[col+1]);
 end;
end;

{write header to printer (header includes: date-label,report title, and page-number}
procedure TcwDBPrint.WriteHeaderToPrinter;
var
 PosX, PosY, t, tmpTitleHeight, li: longint;
 TmpFont: TFont;
 FontCreated: boolean;
begin

 if (tmpPageNo >= FFromPage) and (tmpPageNo <= FToPage) then
  begin
   tmpTitleHeight := TitleHeight;

   with Printer.Canvas do
    begin
     if (FHeader <> '') or (FDatePos <> pnNone) or (FPageNPos <> pnNone) then
      begin
       tmpFont := TFont.Create;
       tmpFont.Assign(Font);         {store current printer font}
       Font.Assign(FHeaderFont);     {assign header\footer font}
       FontCreated := true;
      end
     else
      FontCreated := false;

     {write date-label}
     if FDatePos <> pnNone then
      begin
       PosX := SetPagePosX(FDatePos, FLeftMargin, Printer.PageWidth - (FLeftMargin + FRightMargin));
       PosY := SetPagePosY(FDatePos, FTopMargin, (Printer.PageHeight - (FBottomMargin + TextHeight('X'))));
       TextOut(PosX, PosY, FDateLabel);
      end;

     {write page-number}
     if FPageNPos <> pnNone then
      begin
       PosX := SetPagePosX(FPageNPos, FLeftMargin, Printer.PageWidth - (FLeftMargin + FRightMargin));
       PosY := SetPagePosY(FPageNPos, FTopMargin, (Printer.PageHeight - (FBottomMargin + TextHeight('X'))));
       TextOut(PosX, PosY, FPageNLabel + IntToStr(tmpPageNo));
      end;

     {write header}
     if FHeader <> '' then
      begin
       Font.Assign(FTitleFont); {assign title font}
       PosX := SetAlign(FHeaderAlign, FLeftMargin, Printer.PageWidth - (FLeftMargin + FRightMargin));
       TextOut(PosX, (FTopMargin - TextHeight('X')), FHeader);
      end;

     {set vertical position for records}
     if (FHeader <> '') or (FDatePos in [pnTopLeft, pnTopCenter, pnTopRight])
      or (FPageNPos in [pnTopLeft, pnTopCenter, pnTopRight]) then
       FirstRecordY := (FTopMargin + Scale(TextHeight('M'), FVertGap) + tmpTitleHeight)
     else
      FirstRecordY := FTopMargin + tmpTitleHeight;


     {find extent of lines to be drawn}
     li := FirstRecordY + (FTable.recordCount * LinesHeight) + (FVertGap div 2);
     if li > (printer.pageHeight - (FBottomMargin + FTopMargin)) then
      li := (printer.pageHeight - (FBottomMargin + FTopMargin));

     {set border - note the extent of vertical lines must take into account the record-number
      (plus the fieldname), the vertical spacing, and of course the actual extent of the page}
     if FBorder then
      begin
       {draw border}
       Rectangle(FLeftMargin, FirstRecordY - tmpTitleHeight,
                 FLeftMargin + Positions[NPositions + 1],li);
      end;

     {column seperations}
     if FColLines then
      with Printer.Canvas do
       for t := 2 to NPositions do
        begin
         MoveTo(FLeftMargin + Positions[t], FirstRecordY );
         LineTo(FLeftMargin + Positions[t], li);
        end;

     {free resources for font}
     if FontCreated then
      begin
       Font.Assign(tmpFont);  {restore printer's font}
       tmpFont.Free;
      end;

   end;

  WriteLabelToPrinter(FirstRecordY - tmpTitleHeight); {custom method - writes field-names}

 end;

end;


{write field-names to printer}
procedure TcwDBPrint.WriteLabelToPrinter(PosY: longint);
var
 Col, PosX, t: longint;
 TmpFont: TFont;
 R: TRect;
 pStr: pChar;
begin


 with FTable do
  with Printer.Canvas do
   begin

    {intialise font}
    tmpFont := TFont.Create;
    tmpFont.Assign(Font);
    Font.Assign(FFieldFont);

    {set positioning}
    Col := 0;
    R.top := CenterY(PosY, TextHeight('M'), FVertGap) + (FVertGap div 2); {note: PosY passed to procedure}
    R.bottom := FirstRecordY - FVertGap;

    {iterate through fields, writing field-names to canvas}
    for t := 0 to FieldCount - 1 do
     begin
      if Fields[t].Visible then
       begin
        inc(Col);
	PosX := FLeftMargin + PrepareAlign(Fields[t], Col, true);
	R.left := FLeftMargin + Positions[Col] + FHorizGap;
	R.right := FLeftMargin + Positions[Col+1] - FHorizGap;
	TextRect(R, PosX, R.top, Fields[t].DisplayLabel); {write label}
       end;
     end;

    if FRowLines then
     begin
      MoveTo(FLeftMargin, FirstRecordY);
      LineTo(FLeftMargin + Positions[NPositions + 1], FirstRecordY);
     end;

    Font.Assign(tmpFont);
    tmpFont.Free;

   end;


end;


{write table record to printer}
procedure TcwDBPrint.WriteRecordToPrinter;
var
 Col, t, PosX, PosY: longint;
 tmpFont: TFont;
 R: TRect;
begin
 if (tmpPageNo >= FFromPage) and (tmpPageNo <= FToPage) then
  begin
   with FTable do
    begin
     with Printer.Canvas do
      begin

       {initialise font for printer}
       tmpFont := TFont.Create;
       tmpFont.Assign(Font);
       Font.Assign(FLinesFont);

       {set positioning}
       Col := 0;
       PosY := FirstRecordY + RecCounter * LinesHeight; {vertical position}
       R.top := CenterY(PosY, TextHeight('X'), FVertGap); {central vertical position}
       R.bottom := FirstRecordY + ((RecCounter + 1) * LinesHeight); {bottom of text region}

       {iterate through fields, writing text to canvas}
       for t := 0 to FieldCount - 1 do
        begin
	 if Fields[t].Visible then
	  begin
	   inc(Col);
	   PosX := FLeftMargin + PrepareAlign(Fields[t], Col, false);
	   R.left := FLeftMargin + Positions[Col] + FHorizGap;
	   R.right := FLeftMargin + Positions[Col+1] - FHorizGap;
	   TextRect(R, PosX, R.top, Fields[t].DisplayText); {write text}
          end;

        end;

       {draw row line}
       if FRowLines then
        with Printer.Canvas do
         begin
	  MoveTo(FLeftMargin, PosY);
	  LineTo(FLeftMargin + Positions[NPositions + 1], PosY);
         end;

       Font.Assign(tmpFont);
       tmpFont.Free;

     end;
    end;
  end;
end;


{write header}
procedure TcwDBPrint.WriteHeader;
var
 li: longint;
 s: string;
begin
 if not FToPrint then
  WriteHeaderToPrinter {custom method - handles printer}
 else
  with FTable do
   begin
    WriteLineScreen(FHeader); {write header to file}
    s := '';
    for li := 0 to FieldCount - 1 do
     begin
      if Fields[li].Visible then s := s + Fields[li].DisplayLabel + #9; {#9 = tab}
     end;
    WriteLineScreen(s); {write field-names to file}
   end;
end;


{write record - generic handler of printing dataset's records}
procedure TcwDBPrint.WriteRecord;
var
 w: word;
 s: string;
begin
 if not FToPrint then
  WriteRecordToPrinter {custom method - handles printer}
 else
  begin
   {write to file}
   with FTable do
    begin
     s := '';
     for w := 0 to FieldCount - 1 do
      begin
       if Fields[w].Visible then
        s := s + Fields[w].DisplayText + #9; {#9 = tab}
      end;
    end;
   WriteLineScreen(s); {write field value to file}
  end;
end;


{page jump - move to a fresh page (called by Print)}
procedure TcwDBPrint.PageJump;
begin
 RecCounter := 0; {reset record-counter, forcing header to be printed on new pages}
 if not FToPrint then
  if (tmpPageNo >= FFromPage) and (tmpPageNo < FToPage) then Printer.NewPage;
 inc(tmpPageNo);
end;


{page filled - returns true if page filled with data}
function TcwDBPrint.AllPageFilled: boolean;
begin
 {in case of printing to file, check if record-counter has reached 66, whilst
  in case of printing to printer, check if we are greater than available printing
  area}
 Result := (FToPrint and (RecCounter = 66)) or
           (not FToPrint and ((FirstRecordY + (RecCounter + 1) * LinesHeight)
            >= (Printer.PageHeight - FBottomMargin)));
end;


{***execution methods**********************************************************}

{print}
procedure TcwDBPrint.Print;
var
 res: boolean;
 St: array[0..255] of Char;
 BookMark: TBookMark;
 t: integer;
 tmpFont: TFont;
begin

 {check that grid is assigned}
 if not Assigned(FTable) then
  raise Exception.Create('Dataset property was not specified.');

 {check if printing to file}
 if FToPrint then
  res := OpenTextForWrite
 else
  begin
   res := true;
   with Printer do
    begin
     Title := FPrintMgrTitle;
     BeginDoc; {initialise printer}
     with Canvas do
      begin
       tmpFont := TFont.Create;
       tmpFont.Assign(Font);
       Font.Assign(FLinesFont);
       LinesHeight := Scale(TextHeight('X'), FVertGap);
       LinesWidth := TextWidth('X');
       Font.Assign(tmpFont);
       tmpFont.Free;
      end;
    end;
  end;

 {if ok, then start printing}
 if res then
  begin
    with FTable do
     try
      Screen.Cursor := crHourGlass;
      Bookmark := GetBookMark; {store position in dataset}
      DisableControls;         {disable controls for speed and aesthetics}
      First;
      RecCounter := 0;
      tmpPageNo := 1;
      CalculatePositions; { where to place each field in horizontal plane? }
      if not FToPrint and (Positions[NPositions + 1] > RealWidth) then
       begin
        Screen.Cursor := crDefault;
        MessageDlg('Report width is greater than paper width.',mtWarning,[mbOK],0);
	Screen.Cursor := crHourGlass;
       end;
      {iterate through dataset, printing records}
      while not eof do
       begin
        if RecCounter = 0 then WriteHeader; {on fresh page add header}
        WriteRecord;
	Inc(RecCounter);
 	next; {iterate}
        application.processMessages; {allow Windows to breathe}
        if AllPageFilled then
         begin
          PageJump; {move to next page}
          if tmpPageNo > FToPage then break; {exit if beyond max page}
         end;
       end;
     {cleanup}
     finally
      Screen.Cursor := crDefault;
      GotoBookMark(BookMark);
      EnableControls;
      FreeBookMark(BookMark);
      if FToPrint then
       System.closefile(tmpFile)
      else
       Printer.EndDoc; {force printer to print}
     end;
   end
  else
   raise Exception.Create('Error creating report.');
end;


{show print dialog}
procedure TcwDBPrint.PrintDialog;
begin
 with TPrintDialog.Create(Self) do
  begin
   try

    {set fields}
    Options := [poPageNums, poPrintToFile, poWarning];	{poHelp}
    MinPage := 1;
    MaxPage := MaxPages;
    FFromPage := 1;
    FToPage := MaxPages;

    {on ok, print!}
    if Execute then
     begin
      if PrintRange = prPageNums then
       begin
        FFromPage := FromPage;
        FToPage := ToPage;
       end;
      if not PrintToFile then
       begin
       {printing}
        FToPrint := false;
        Print;
       end
      else
       {writing to file}
       begin
        FToPrint := true;
        with TSaveDialog.Create(Self) do
         begin
          try
           Filter := 'Text files (*.TXT)|*.TXT|Any file (*.*)|*.*';
           if FPrintFileName <> '' then
            begin
             {set fields for TSaveDialog}
	     FileName := FPrintFileName;
	     Filter := Filter + '|This file (*' + ExtractFileExt(FileName) + ')|*' + ExtractFileExt(FileName);
	     FilterIndex := 3;
            end;
           if Execute then
            begin
             if FileExists(FileName) then
              if MessageDlg(FileName + ' already exists. Do you wish to overwrite this file?',mtConfirmation,[mbYes,mbNo],0)
                = mrYes then
                 begin
                  tmpFileName := FileName;
   	          Print;
                 end;
            end;
          finally
           Free;
          end;

         end;
       end;
     end;
  finally
   Free;
  end;
 end;
end;



{}
end.
